/*
Half-Life MAP viewing utility.
Copyright (C) 2003  Ryan Samuel Gregg

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#pragma once
#include "stdafx.h"
#include "WorldObject.h"
#include "TextureManager.h"

__gc class CFace : public CWorldObject
{
private:
	bool bSpecialTexture;
	
	int iIndex;
	unsigned int iID;
	Vector *vAxisU;
	Vector *vAxisV;
	Vertex2f vShift;
	Vertex2f vScale;
	float fRotation;
	
	int iNumVertices;

	Vertex2f *TexCoords;
	Vertex3f *Vertices;
	Vertex3f vNormal;

public:
	CFace(CConfig *Config, int iNumVertices) : CWorldObject(Config)
	{
		bSpecialTexture = false;
		this->iNumVertices = iNumVertices;

		TexCoords = new Vertex2f[iNumVertices];
		Vertices = new Vertex3f[iNumVertices];
	}

	~CFace()
	{
		delete []TexCoords;
		delete []Vertices;
	}
	
	void SetVertex(int iIndex, Vertex3f Vertex)
	{
		Vertices[iIndex] = Vertex;
	}

	void SetNormal(Vertex3f vNormal)
	{
		this->vNormal.X = vNormal.X;
		this->vNormal.Y = vNormal.Y;
		this->vNormal.Z = vNormal.Z;
	}

	void SetSpecialTexture(bool bSpecialTexture)
	{
		this->bSpecialTexture = bSpecialTexture;
	}

	void SetTextureInfo(int iIndex, Vector *vAxisU, Vector *vAxisV, Vertex2f vShift, Vertex2f vScale, float fRotation)
	{
		this->iIndex = iIndex;
		this->vAxisU = vAxisU;
		this->vAxisV = vAxisV;
		this->vShift.X = vShift.X;
		this->vShift.Y = vShift.Y;
		this->vScale.X = vScale.X;
		this->vScale.Y = vScale.Y;
		this->fRotation = fRotation;
	}

	void SetTextureID(unsigned int iID)
	{
		this->iID = iID;
	}

	Vertex3f *GetVertices()
	{
		return Vertices;
	}

	Vertex3f GetVertex(int iIndex)
	{
		return Vertices[iIndex];
	}
	
	int GetVertexCount()
	{
		return iNumVertices;
	}

	String *GetTexture(CTextureManager *TextureManager)
	{
		return TextureManager->GetName(iIndex);
	}

	float GetUShift()
	{
		return vShift.X;
	}

	float GetVShift()
	{
		return vShift.Y;
	}

	float GetUScale()
	{
		return vScale.X;
	}

	float GetVScale()
	{
		return vScale.Y;
	}

	float GetRotation()
	{
		return fRotation;
	}

	void UpdateTexCoords(CTextureManager *TextureManager)
	{
		iID = TextureManager->GetID(iIndex);

		float W, H, SX, SY;
		W = 1.0f / (float)TextureManager->GetWidth(iIndex);
		H = 1.0f / (float)TextureManager->GetHeight(iIndex);
		SX = 1.0f / vScale.X;
		SY = 1.0f / vScale.Y;

		Vector *V;
		for(int i = 0; i < iNumVertices; i++)
		{
			V = new Vector(Vertices[i]);

			TexCoords[i].X = (V->Dot(vAxisU) * W * SX) + (vShift.X * W);
			TexCoords[i].Y = (V->Dot(vAxisV) * H * SY) + (vShift.Y * H);
		}
	}

	void DrawFaceTextured()
	{
		if(bSpecialTexture && !Config->bDrawSpecialTextures)
			return;

		glBindTexture(GL_TEXTURE_2D, iID);

		glBegin(GL_POLYGON);
		glNormal3f(vNormal.X, vNormal.Y, vNormal.Z);
		for(int i = 0; i < iNumVertices; i++)
		{
			glTexCoord2fv(&TexCoords[i].X);
			glVertex3fv(&Vertices[i].X);
		}
		glEnd();
	}

	void DrawFaceSolid()
	{
		if(bSpecialTexture && !Config->bDrawSpecialTextures)
			return;

		glBegin(GL_POLYGON);
		glNormal3f(vNormal.X, vNormal.Y, vNormal.Z);
		for(int i = 0; i < iNumVertices; i++)
		{
			glVertex3fv(&Vertices[i].X);
		}
		glEnd();
	}

	void DrawFaceWireFrame()
	{
		if(bSpecialTexture && !Config->bDrawSpecialTextures)
			return;

		glBegin(GL_LINE_LOOP);
		for(int i = 0; i < iNumVertices; i++)
		{
			glVertex3fv(&Vertices[i].X);
		}
		glEnd();
	}

	void DrawFacePoints()
	{
		if(bSpecialTexture && !Config->bDrawSpecialTextures)
			return;

		glBegin(GL_POINTS);
		for(int i = 0; i < iNumVertices; i++)
		{
			glVertex3fv(&Vertices[i].X);
		}
		glEnd();
	}

	void Highlight(bool bPrepared)
	{
		if(!bPrepared)
		{
			PrepareHighlight();
		}

		glBegin(GL_LINE_LOOP);
		for(int i = 0; i < iNumVertices; i++)
		{
			glVertex3fv(&Vertices[i].X);
		}
		glEnd();
	}

	void Outline()
	{
		if(bSpecialTexture && !Config->bDrawSpecialTextures)
			return;

		glBegin(GL_LINE_LOOP);
		for(int i = 0; i < iNumVertices; i++)
		{
			glVertex3fv(&Vertices[i].X);
		}
		glEnd();
	}
};